/** @file   gameobject.h
 * @brief   Declaration of GameObject - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_GAMEOBJECT_H
#define H_WWW_GAMEOBJECT_H

#include <vector>
#include "eng2d.h"
#include "gameanims.h"
#include "animatedobject.h"
#include "objectid.h"
#include "bullettable.h"


namespace WeWantWar {

// Just let the compiler know about these classes.
class RedrawQueue;
class BaseController;
class ObjectMessage;


/** @class  GameObject
 * @brief   Superclass of all the objects that can interact with each other.
 * @author  Tomi Lamminsaari
 *
 * This class manages the state and status information of the objects such
 * as current health- and armor levels.
 *
 * This class also describes the interface for updating and drawing
 * purposes. The child classes must implement this interface.
 *
 * There is an IDCode attribute in this class. It can have an arbitary value
 * but by default all the GameObjects have GameObject::DEFAULT_ID as their
 * id-code. This attribute is needed for instance to distinguish GameObjects
 * from each other when joining them to a Team.
 *
 * @par Interaction with game itself.
 * The main loop of the game calls the <code> updateObject() </code> -
 * method about 40 times per second. This method then calls the
 * <code> update() </code> - method that must be implemented in the
 * child classes.
 * @par
 * When it's time to draw the object, the main control loop of the game calls
 * <code> redraw( BITMAP* pTarget ) </code> - method. This method must
 * be implemented in child classes. That method should draw this object on
 * provided bitmap.
 */
class GameObject : public AnimatedObject
{
public:
  /** Static members and methods **/
  
  /** Number of extra counters the instances of GameObject-class has. */
  static const int NUM_OF_COUNTERS = 3;
  
  /** Number of collision points the GameObjects have.
   */
  static const int NUM_OF_COLL_POINTS = 4;
  
  /** During the gameplay this tells how many instances of GameObject we have
   * created. It is incremented in constructor and decremented in destructor.
   */
  static int s_numberOfGameObjects;
  
  
  /** The states the GameObjects can be in */
  enum State {
    /** Object is living */
    STATE_LIVING,
    /** Object does not respond any controlling commands and it's ignored
     * in collision control.
     */
    STATE_HIBERNATING,
    /** This object has been killed */
    STATE_KILLED,
    /** This object is dying ( running the dying animation ) */
    STATE_DYING
  };
  
  
  /** These are the standard sounds the objects make. The @c makeSound(...) -
   * method should be capable of producing these sounds.
   */
  enum SoundID {
    /** Idle sound. This is played every now and then */
    SND_IDLE,
    /** Object got hurt. */
    SND_PAIN,
    /** This object attacks or shoot */
    SND_ATTACK,
    /** Object is walking or moving. */
    SND_WALK,
    /** Object dies. */
    SND_DIE,
    /** Sound the object makes when it gets attracted by another object. */
    SND_WAKE,
    /** Reloading sound */
    SND_RELOAD
  };


  /** Datatype for properymask. */
  typedef int BProp;
  /** Bitmask for property that tells if this object is affected by
   * the force caused by grenades.
   */
  static const BProp PROP_GRENADEFORCE = 1;
  /** Bitmask for property that tells, can this object attack while he
   * himself is attacked. Usually this flag is used only by shooting
   * aliens who can't shoot back while they're being shot.
   */
  static const BProp PROP_SHOTPARALYSES = 2;
  
  
  
  /** Datatype for GameObject ID-code. */
  typedef int IDCode;
  /** Default ID-code for the objects. */
  static const IDCode DEFAULT_ID = 0;
  


  ///
  /// Constructors, destructor and operators
  /// ==========================================

  /** Constructs new gameobject. This constructor sets the following values
   * for the attributes:
   * - Hidden flag = false
   * - Health = 100
   * - Armor = 1
   * - Proprtyflags = none
   * - State = STATE_LIVING
   * - IDCode = DEFAULT_ID
   * - 4 collisionpoints are made. They all are located at (0,0)
   */
  GameObject();
  
  /** Destructor. This will delete the Controller - object.
   */
  virtual ~GameObject();


  

  ///
  /// Methods
  /// =======

  
  /** Tells the target of this object. Default implementation does nothing.
   * Very likely this method will be removed in future.
   * @param     t                 Vector that points the target coordinate.
   */
  virtual void target( const eng2d::Vec2D& t );
  
  
  /** Sets the state of this object. If state == STATE_KILLED this object
   * will be ignored in collision checks.
   * @param     s                 New state.
   */
  virtual void state(State s);
  
  /** Sets the counter-values. Each GameObject has 3 counters that can be
   * used for any purpose the descedant-class wants. The counters count
   * downwards.
   * @param     cnum              Index of the counter (0, 1, 2)
   * @param     val               New value for the counter.
   */
  void setCounter( int cnum, int val );
  
  /** Sets the controlpoint. Each GameObject can have any number of
   * so called control points. They are rotated as the object rotates.
   * For instance the Player - object uses one controlpoint for the "nose"
   * of the weapon.
   *
   * The controlpoints will be rotated around the origin so they should be
   * relative to the object's position.
   * @param     pnum              Index of the controlpoints
   * @param     p                 The controlpoint
   */
  void setCtrlPoint( int pnum, const eng2d::Vec2D& p );
  
  /** Adds new control point. Each GameObject can have any number of
   * so called control points. They are rotated as the object rotates.
   * For instance the Player - object uses one controlpoint for the "nose"
   * of the weapon.
   *
   * The controlpoints will be rotated around the origin so they should be
   * relative to the object's position.
   * @param     p                 The point being added.
   */
  void addCtrlPoint( const eng2d::Vec2D& p );
  
  /** Sets the 'hidden'-flag on / off. If hidden-flag is on, this object is
   * ignored by the redrawing code.
   * @param     h                 Hides or shows this object.
   */
  void hidden(bool h);
  
  
  
  /** Updates this GameObject. This is pure virtual method and therefore
   * every child classes has to implement it. This method is called from
   * <code>updateObject()</code> - method.
   */
  virtual void update() = 0;
  
  /** This is the main update-method. This is called from the game itself
   * and this updates the counters and other properties the GameObject has.
   *
   * This calls the <code>update</code> - method and it should update the
   * child-class.
   */
  void updateObject();
  
  
  /** Redraws this gameobject. This method must be implemented in child
   * classes.
   * @param     pQueue            Pointer to redraw queue.
   */
  virtual void redraw( RedrawQueue* pQueue ) = 0;
  
  /** Causes given points of damage to this GameObject.
   * @param     points            how many damage points.
   * @return    'true' if this GameObject just died.
   */
  virtual bool causeDamage( int points );
  
  /** Causes the damage by bullethit.
   * @param     b                 Pointer to Bullet that hit us.
   * @return    'true' if this GameObject just died.
   */
  virtual bool causeDamage( Bullet* b );
  
  /** This object was hit by given bullet. This method can either accept the
   * bullethit or reject it. If it accepts the hit, it should call the
   * <code>causeDamage( Bullet* b )</code> - method and pass the given
   * Bullet - parameter to it. This method should return @c true if it
   * accepts the bullethit. If it rejects the hit, it should return @c false .
   *
   * This method should always check if we're shooting ourselves. This case
   * should not be accepted.
   *
   * This method should also spawn the possible bullet-hit animations.
   * @param     pB                Pointer to bullet that hit us.
   * @return    'true' if this object was hit. 'false' means that the
   *            bullet went through.
   */
  virtual bool hitByBullet( Bullet* pB ) = 0;
  
  /** Kills this object. This method should change our state to @c STATE_DYING
   * or @c STATE_KILLED depending on how the object should react. Also this
   * method should call the <code>makeSound( SoundID id )</code> - method
   * and start the dying animation.
   */
  virtual void kill() = 0;
  
  /** Makes sound. This must by implemented in the descedant-classes.
   * @param     id                ID of the sound this GameObject should make.
   */
  virtual void makeSound( SoundID id ) const = 0;


  /** Sets the health level.
   * @param     h                 The health level. 0 = dead, 100 = default
   */
  void setHealth( int h );
  
  /** Sets the armor-level. When bullethit occures, the damage-points are
   * divided by armor-level and our health-level will be reduced by that
   * quatient.
   * @param     a                 Armor-level.
   */
  void setArmor( float a );
  
  /** Sets the properties. The properties should be or'd together.
   * @param     p                 New set of properties
   */
  void setProperties( BProp p );
  
  /** Sets the idcode. These ID-codes are needed when we're grouping several
   * GameObjects as a team. We need the ID-codes also when creating Dynamic
   * Map Effects that can be activated by a certain GameObject only.
   *
   * There are no predefined ID-codes. By default all the GameObjects will
   * have @c GameObject::DEFAULT_ID
   * @param     newID             New  IDCode.
   */
  void objectID( IDCode newID );
  
  /** Sets the controller. The ownership of the given Controller - object will
   * be moved to us. We are respossible of deleting it when it comes useless.
   * @param     pController       Pointer to our controller.
   */
  void setController( BaseController* pController );
  
  /** Wakes this object from death.
   */
  virtual void resurrect();
  
  /** The messages from other GameObjects are sent to us through this
   * method. This method sends a response message back to the object
   * that sent the message.
   *
   * @note      If the given message is a response-message, you must not
   *            send a response to it.
   * @param     rMessage          Reference to the message
   */
  virtual void messagePort( const ObjectMessage& rMessage );
  
  

  ///
  /// Getter methods
  /// ==============
  
  
  /** Returns the controlpoints. This method will rotate the controlpoint first
   * and returns the rotated version.
   * @param     pnum              Index of the controlpoint being returned.
   * @return    The rotated control.
   */
  eng2d::Vec2D  getCtrlPoint(int pnum) const;
  
  
  /** Returns the type of this object.
   * @return    Type of this object.
   */
  virtual ObjectID::Type  objectType() const = 0;
  
  /** Returns the idcode of this GameObject.
   * @return    The idcode.
   */
  IDCode objectID() const;
  
  /** Returns the current state of this object.
   * @return    Current state.
   */
  State         state() const;
  
  /** Returns the value of certain counter.
   * @param     cnum              Index of the counter (0, 1, 2)
   * @return    Counter's value.
   */
  int           getCounter(int cnum) const;
  
  /** Is this object hidden.
   * @return    'true' if hidden
   */
  bool          hidden() const;
  
  
  /** Returns the health-level.
   * @return    Health-level. 0=should be dead, 100 = ok
   */
  int           health() const;
  
  /** Returns the armor-level
   * @return    Armor-level
   */
  float         armor() const;

  /** Obsoled.
   * @return    Always 'false'
   */
  virtual bool  playerInSight() const {return false; }
  
  /** Is this object currently reloading its weapon.
   * @return    'true' if reloading.
   */
  virtual bool  reloading() const = 0;
  
  /** Tells if the given propery is on.
   * @param     p                 A bitmask for requested property
   * @return    <code> true </code> if at least one of the requested
   *            properties is on.
   */
  bool hasProperty( BProp p ) const;
  
  /** Returns all the property flags. You can use the BProp - bitmasks
   * to find out which properties are on and which are off.
   * @return    The propertyflags.
   */
  int getProperties() const;
  
  /** Returns the controller we're using. The caller must not delete this
   * object.
   * @return    Pointer to our controller object
   */
  BaseController* getController() const;

  
protected:

  ///
  /// Members
  /// =======

  /** Condition of this object. 0 = dead, 100 = ok */
  float     m_health;
  
  /** The damage-points get divided by this armor level before they are
   * substracted from health-level.
   */
  float     m_armor;
  
  /** THe current state of this GameObject. */
  State     m_state;
  
  
  /** Force-vector is an extra movement vector that is added to the object's
   * position. After this addition we reduce the magnitude of it so that
   * its effect fades away. We use this for instance to simulate the presure
   * blast caused ny the grenades. Also the collisions between player and
   * aliens are done by giving them extra boost away from each other.
   */
  eng2d::Vec2D  m_force;
  
  
  /** Is this GameObject hidden */
  bool      m_hidden;
  
  /** This gets set to 'true' or 'false' during the each call of update()-
   * method. It's set 'true' when this GameObject touches a door.
   */
  bool      m_touchedWall;
  
  
  /** GameObject-class has some extra counters that can be used for any
   * purpose the descedant-classes want to. The values of these counters
   * gets reduced by 1 during the each call of update()-method.
   */
  int       m_counters[NUM_OF_COUNTERS];
  
  /** GameObjects can have any number of control-points. */
  std::vector<eng2d::Vec2D> m_ctrlPoints;
  
  /** A bitwise properties this object has.
   */
  int       m_propertyFlags;
  
  /** An ID-number of this GameObject. */
  IDCode    m_objectID;
  
  /** The controller used by this object. */
  BaseController* m_pController;
  
private:
  GameObject(const GameObject& rO);
  GameObject& operator = (const GameObject& rO);
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: gameobject.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-23 00:24:00+02  lamminsa
 * <>
 *
 * Revision 1.0  2005-11-06 01:16:58+02  lamminsa
 * Initial revision
 *
 */
 

